<?php
/*--------------------------------------------------------------
   Option.php 2021-06-03
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2021 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
 -------------------------------------------------------------*/

declare(strict_types=1);

namespace Gambio\Admin\Modules\Statistics\App\Overview\Factory;

use Gambio\Admin\Modules\Language\Model\ValueObjects\LanguageCode;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Dropdown as DropdownOptionFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Factory\Option\Predefined as PredefinedOptionFactory;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\Options;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Collections\OptionTitles;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\Entities\WidgetDefinition;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option as OptionInterface;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\Checkbox as CheckboxOption;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\Dropdown as DropdownOption;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\Id;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\Number as NumberOption;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\Text as TextOption;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\Title;
use Gambio\Admin\Modules\Statistics\App\Overview\Model\ValueObjects\Option\UpdateSet;
use InvalidArgumentException;

class Option
{
    /**
     * @var DropdownOptionFactory
     */
    private $dropdownFactory;
    
    /**
     * @var PredefinedOptionFactory
     */
    private $predefinedFactory;
    
    
    /**
     * Constructor.
     */
    public function __construct(DropdownOptionFactory $dropdownFactory, PredefinedOptionFactory $predefinedFactory)
    {
        $this->dropdownFactory   = $dropdownFactory;
        $this->predefinedFactory = $predefinedFactory;
    }
    
    
    /**
     * Return dropdown factory.
     */
    public function useDropdowns(): DropdownOptionFactory
    {
        return $this->dropdownFactory;
    }
    
    
    /**
     * Return predefined options factory.
     */
    public function usePredefined(): PredefinedOptionFactory
    {
        return $this->predefinedFactory;
    }
    
    
    /**
     * Return options.
     */
    public function createOptions(OptionInterface ...$options): Options
    {
        return Options::create(...$options);
    }
    
    
    /**
     * Return title.
     */
    public function createTitle(LanguageCode $languageCode, string $title): Title
    {
        return Title::create($languageCode, $title);
    }
    
    
    /**
     * Return titles.
     */
    public function createTitles(Title ...$titles): OptionTitles
    {
        return OptionTitles::create(...$titles);
    }
    
    
    /**
     * Return number option.
     */
    public function createNumber(Id $id, int $value, OptionTitles $titles): NumberOption
    {
        return NumberOption::create($id, $value, $titles);
    }
    
    
    /**
     * Return text option.
     */
    public function createText(Id $id, string $value, OptionTitles $titles): TextOption
    {
        return TextOption::create($id, $value, $titles);
    }
    
    
    /**
     * Return checkbox option.
     */
    public function createCheckbox(Id $id, bool $value, OptionTitles $titles): CheckboxOption
    {
        return CheckboxOption::create($id, $value, $titles);
    }
    
    
    /**
     * Return update set.
     */
    public function createUpdateSet(array $updateSet): UpdateSet
    {
        return UpdateSet::create($updateSet);
    }
    
    
    /**
     * Return ID.
     */
    public function createId(string $id): Id
    {
        return Id::create($id);
    }
    
    
    /**
     * Return update set from options.
     */
    public function createUpdateSetFromOptions(Options $options): UpdateSet
    {
        $updateSet = [];
        
        /**
         * @var OptionInterface $option
         */
        foreach ($options as $option) {
            $updateSet[$option->id()->value()] = $option->value();
        }
        
        return $this->createUpdateSet($updateSet);
    }
    
    
    /**
     * Return options from update set and widget definition.
     */
    public function createOptionsFromUpdateSet(UpdateSet $updateSet, WidgetDefinition $definition): Options
    {
        return $this->createOptions(...array_map(function (OptionInterface $option) use ($updateSet) {
            $id     = $option->id();
            $value  = $updateSet->toArray()[$id->value()] ?? $option->value();
            $titles = $option->titles();
            
            if ($option instanceof CheckboxOption) {
                return $this->createCheckbox($id, $value, $titles);
            }
            
            if ($option instanceof DropdownOption) {
                return $this->useDropdowns()->createDropdown($id, $value, $option->items(), $titles);
            }
            
            if ($option instanceof NumberOption) {
                return $this->createNumber($id, $value, $titles);
            }
            
            if ($option instanceof TextOption) {
                return $this->createText($id, $value, $titles);
            }
            
            throw new InvalidArgumentException("Option not found");
        },
            iterator_to_array($definition->options()->getIterator())));
    }
}